// =============================================================================
// Eriksson Beam API - Simple Sample
// =============================================================================
// This sample demonstrates the basic workflow for using the Eriksson Beam API:
//   1. Launch Eriksson Beam with a project file
//   2. Pull the current design data
//   3. Modify properties
//   4. Push changes back to Eriksson Beam
//
// Prerequisites:
//   - Eriksson Beam desktop application installed
//   - Valid API license key
//   - A .ebf project file to work with
//
// For more information, see the documentation at:
//   https://green-plant-030f9990f.azurestaticapps.net/
// =============================================================================

using System;
using System.IO;
using System.Threading.Tasks;
using ErikssonBeam.API.BeamClient;

namespace SimpleApiSample
{
    class Program
    {
        static async Task Main(string[] args)
        {
            // =========================================================
            // CONFIGURATION - Update these values for your environment
            // =========================================================

            // Your API license key (contact Eriksson Software to obtain)
            // Tip: For production use, store this in an environment variable
            var licenseKey = "YOUR_LICENSE_KEY_HERE";

            // Path to a beam project file (.ebf)
            var projectFilePath = @"C:\Path\To\Your\Project.ebf";

            // =========================================================
            // STEP 1: Configure the launcher
            // =========================================================
            Console.WriteLine("Configuring Eriksson Beam launcher...");

            var launcherArgs = new ErikssonBeamLauncherArgs
            {
                // Your license key for API access
                LicenseKey = licenseKey,

                // Files to open when Eriksson Beam launches
                FilesToOpen = new[] { projectFilePath },

                // When true, Eriksson Beam closes when we disconnect
                CloseErikssonBeamOnClientDisconnect = true
            };

            // =========================================================
            // STEP 2: Launch Eriksson Beam and connect
            // =========================================================
            Console.WriteLine("Launching Eriksson Beam...");

            // The 'using' statement ensures proper cleanup when done
            using (var launcher = await ErikssonBeamLauncher.LaunchErikssonBeam(launcherArgs))
            {
                // Get the client for API operations
                var client = launcher.Client;
                Console.WriteLine("Connected to Eriksson Beam!");

                // =========================================================
                // STEP 3: Pull the current design data
                // =========================================================
                Console.WriteLine("\nPulling design data...");

                // PullBeamDesignerAsync retrieves all input data from the open project
                // The parameter is the project identifier (filename without extension)
                var projectName = Path.GetFileNameWithoutExtension(projectFilePath);
                var design = await client.PullBeamDesignerAsync(projectName);

                // Display current values
                Console.WriteLine("\n--- Current Values ---");
                Console.WriteLine($"Project Name: {design.DesignCriteria.ProjectInformation.ProjectName}");
                Console.WriteLine($"Designer:     {design.DesignCriteria.ProjectInformation.DesignerName}");
                Console.WriteLine($"Client:       {design.DesignCriteria.ProjectInformation.ClientName}");
                Console.WriteLine($"Date:         {design.DesignCriteria.ProjectInformation.Date}");

                // =========================================================
                // STEP 4: Modify design properties
                // =========================================================
                Console.WriteLine("\nModifying design properties...");

                // Modify project information
                design.DesignCriteria.ProjectInformation.DesignerName = "API Sample User";
                design.DesignCriteria.ProjectInformation.ClientName = "Sample Client";
                design.DesignCriteria.ProjectInformation.Date = DateTime.Now;

                // TIP: Always batch your modifications before pushing!
                // Each push is an expensive operation, so make all changes
                // to the design object first, then push once.

                // =========================================================
                // STEP 5: Push changes back to Eriksson Beam
                // =========================================================
                Console.WriteLine("Pushing changes to Eriksson Beam...");

                // PushBeamDesignerAsync sends all modifications back to the application
                await client.PushBeamDesignerAsync(projectName, design);

                Console.WriteLine("\n--- Updated Values ---");
                Console.WriteLine($"Designer:     {design.DesignCriteria.ProjectInformation.DesignerName}");
                Console.WriteLine($"Client:       {design.DesignCriteria.ProjectInformation.ClientName}");
                Console.WriteLine($"Date:         {design.DesignCriteria.ProjectInformation.Date}");

                Console.WriteLine("\nSuccess! Changes have been pushed to Eriksson Beam.");
                Console.WriteLine("Check the Eriksson Beam window to verify the updates.");

                // =========================================================
                // CLEANUP
                // =========================================================
                // The 'using' block automatically disposes the launcher,
                // which closes the client connection and (because we set
                // CloseErikssonBeamOnClientDisconnect = true) also closes
                // Eriksson Beam itself.
            }

            Console.WriteLine("\nPress any key to exit...");
            Console.ReadKey();
        }
    }
}
